/**
* \file: signal_handler.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADITG / ESM / mhoyer@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <signal.h>
#include <unistd.h>
#include <sys/signalfd.h>

#include "utils/logger.h"
#include "control/automounter.h"
#include "control/watch.h"
#include "control/signal_handler.h"

static void signal_handler_on_event(watch_t *ev_watch, uint32_t events);

static watch_t sigfd_watch = {-1, 0, signal_handler_on_event, NULL};
static sigset_t sigset;
static bool isdaemon = false;

error_code_t signal_handler_init(void)
{
    int sig_file_descriptor;

    sigemptyset(&sigset);

    if (sigaddset(&sigset, SIGTERM) == -1)
    {
	    logger_log_error("SIGNAL_HANDLER - Unable to add SIGTERM to sigset.");
            return RESULT_NORESOURCE;
    }
    if (sigaddset(&sigset, SIGINT) == -1)
    {
	    logger_log_error("SIGNAL_HANDLER - Unable to add SIGINT to sigset.");
            return RESULT_NORESOURCE;
    }
    if (sigaddset(&sigset, SIGHUP) == -1)
    {
	    logger_log_error("SIGNAL_HANDLER - Unable to add SIGHUP to sigset.");
            return RESULT_NORESOURCE;
    }

    if (sigprocmask(SIG_BLOCK, &sigset, NULL) == -1)
    {
	    logger_log_error("SIGNAL_HANDLER - Unable to block signals.");
	    return RESULT_NORESOURCE;
    }

    sig_file_descriptor = signalfd(sigfd_watch.pollfd, &sigset, 0);
    if (sig_file_descriptor == -1)
    {
	    logger_log_error("SIGNAL_HANDLER - Unable to create signal file descriptor.");
	    return RESULT_NORESOURCE;
    }
    logger_log_debug("SIGNAL_HANDLER - Signal file descriptor %d created.", sig_file_descriptor);

    sigfd_watch.pollfd = sig_file_descriptor;

    watch_add_event_source(&sigfd_watch, EPOLLIN);

    return RESULT_OK;
}

error_code_t signal_handler_deinit(void)
{
    if (sigfd_watch.pollfd!=-1)
    {
    	    (void)watch_remove_event_source(&sigfd_watch);
            close(sigfd_watch.pollfd);
            sigfd_watch.pollfd=-1;

            sigprocmask(SIG_UNBLOCK, &sigset, NULL);
    }

    return RESULT_OK;
}

error_code_t signal_handler_refresh(void)
{
    int sig_file_descriptor;

    isdaemon = true;

    sig_file_descriptor = signalfd(sigfd_watch.pollfd, &sigset, 0);
    if (sig_file_descriptor == -1)
    {
	    logger_log_error("SIGNAL_HANDLER - Unable to create signal file descriptor.");
	    return RESULT_NORESOURCE;
    }
    logger_log_debug("SIGNAL_HANDLER - Signal file descriptor %d refreshed.", sig_file_descriptor);

    (void)watch_remove_event_source(&sigfd_watch);
    sigfd_watch.pollfd = sig_file_descriptor;
    watch_add_event_source(&sigfd_watch, EPOLLIN);

    return RESULT_OK;
}

//------------------------------------------ private members ---------------------------------------------
static void signal_handler_on_event(watch_t *ev_watch, uint32_t events)
{
    struct signalfd_siginfo siginfo;
    ssize_t res;

    (void)ev_watch;
    (void)events;

    if (sigfd_watch.pollfd != -1)
    {
            res = read(sigfd_watch.pollfd, &siginfo, sizeof(siginfo));
            if (res != sizeof(siginfo))
            {
                    logger_log_error("SIGNAL_HANDLER - Unable to process signal.");
    	            return;
            }

            if (siginfo.ssi_signo == SIGINT || siginfo.ssi_signo == SIGTERM)
            {
                    logger_log_debug("SIGNAL_HANDLER - Caught signal %d.", siginfo.ssi_signo);
                    automounter_term();
    	    }
            else if (siginfo.ssi_signo == SIGHUP)
            {
        	    if (!isdaemon)
        	    {
        		    logger_log_debug("SIGNAL_HANDLER - Caught signal %d.", siginfo.ssi_signo);
        		    automounter_term();
        	    }
        	    else
        	    {
        		    logger_log_debug("SIGNAL_HANDLER - Ignored signal %d as daemon.", siginfo.ssi_signo);
        	    }
            }
    	    else
    	    {
    		    logger_log_error("SIGNAL_HANDLER - Caught unexpected signal %d.", siginfo.ssi_signo);
    	    }
    }
}
